/*
 * Panasonic EHCI Host Controller Driver
 *
 * Based on "ehci-ixp4xx.c" by Vladimir Barinov <vbarinov@embeddedalley.com>
 *
 * Copyright (C) 2010, Panasonic Corporation 
 *
 * This file is licensed under
 * the terms of the GNU General Public License version 2. This program
 * is licensed "as is" without any warranty of any kind, whether express
 * or implied.
 */

#include <linux/platform_device.h>

static int panasonic_ehci_setup( struct usb_hcd *hcd )
{
	struct ehci_hcd		*ehci = hcd_to_ehci(hcd);
	int					retval;
	
	ehci->caps = hcd->regs + INTERNAL_EHCI_CAPS_BASE;
	ehci->regs = hcd->regs + INTERNAL_EHCI_REGS_BASE;
	
	/* cache this readonly data; minimize chip reads */
	ehci->hcs_params = ehci_readl( ehci, &ehci->caps->hcs_params );
	
	retval = ehci_halt(ehci);
	if( retval ) return retval;
	
	/* data structure init */
	retval = ehci_init(hcd);
	if( retval ) return retval;
	
	/* BURSTSIZE */
	_WRITEL ( USBH_EHCI_BURSTSIZE, &ehci->regs->burstsize );
	/* TXFILLTUNING */
	_WRITEL ( USBH_EHCI_TXFILLTUNING, &ehci->regs->txfilltuning );
	
	hcd->has_tt = 1;
	tdi_reset(ehci);
	
	ehci_reset(ehci);
	
	ehci_port_power( ehci, 0 );
	
	return retval;
}

static const struct hc_driver panasonic_ehci_hc_driver = {
	.description		= hcd_name,
	.product_desc		= "panasonic EHCI Host Controller",
	.hcd_priv_size		= sizeof(struct ehci_hcd),
	.irq				= ehci_irq,
	.flags				= HCD_MEMORY | HCD_USB2,
	.reset				= panasonic_ehci_setup,
	.start				= ehci_run,
	.stop				= ehci_stop,
	.shutdown			= ehci_shutdown,
	.urb_enqueue		= ehci_urb_enqueue,
	.urb_dequeue		= ehci_urb_dequeue,
	.endpoint_disable	= ehci_endpoint_disable,
	.endpoint_reset		= ehci_endpoint_reset,
	.get_frame_number	= ehci_get_frame,
	.hub_status_data	= ehci_hub_status_data,
	.hub_control		= ehci_hub_control,
#if defined(CONFIG_PM)
	.bus_suspend		= ehci_bus_suspend,
	.bus_resume			= ehci_bus_resume,
#endif
	.relinquish_port	= ehci_relinquish_port,
	.port_handed_over	= ehci_port_handed_over,
	
	.clear_tt_buffer_complete	= ehci_clear_tt_buffer_complete,
};

static int panasonic_ehci_probe(struct platform_device *pdev)
{
	const struct hc_driver *driver = &panasonic_ehci_hc_driver;
	struct usb_hcd	*hcd;
	struct resource *res;
	unsigned long	rsrc_start;
	unsigned long	rsrc_len;
	int		irq;
	int		retval=0;
	
	if( usb_disabled() ) return -ENODEV;

	res = platform_get_resource( pdev, IORESOURCE_IRQ, 0 );
	assert( res );
	if( !res ) goto done;
	irq = res->start;

	res = platform_get_resource( pdev, IORESOURCE_MEM, 0 );
	assert( res );
	if( !res ) goto done;
	rsrc_start = res->start;
	rsrc_len =   resource_size(res);

	hcd = usb_create_hcd( driver, &pdev->dev, dev_name(&pdev->dev) );
	if( !hcd ){
		retval = -ENOMEM;
		goto done;
	}

	hcd->rsrc_start = rsrc_start;
	hcd->rsrc_len = rsrc_len;
	hcd->regs = (struct ehci_regs*)( rsrc_start - USBH_IOBASEADDR );

#if defined(CONFIG_USB_PANASONIC_SLD2H) || defined(CONFIG_USB_PANASONIC_SLD3) || defined(CONFIG_USB_PANASONIC_LD4)

	hcd->regs = devm_ioremap_nocache( &pdev->dev, (resource_size_t)hcd->regs, rsrc_len );
#endif
	hcd->irq = irq;

#ifdef CONFIG_SS_SC

	retval = usb_add_hcd( hcd, irq, (IRQF_SHARED | IRQF_SAMPLE_RANDOM) );
#else

	retval = usb_add_hcd( hcd, irq, IRQF_SHARED );
#endif

	if( retval ){
		usb_put_hcd( hcd );
		goto done;
	}
	
done:
	return retval;
}

static int panasonic_ehci_remove(struct platform_device *pdev)
{
	struct usb_hcd *hcd = platform_get_drvdata(pdev);
	
	usb_remove_hcd(hcd);
	usb_put_hcd(hcd);
	
	return 0;
}

MODULE_ALIAS("platform:panasonic-ehci");

static struct platform_driver panasonic_ehci_driver = {
	.probe = panasonic_ehci_probe,
	.remove = panasonic_ehci_remove,
	.driver = {
		.name = "panasonic-ehci",
	},
};

#define HOST_NUM	ARRAY_SIZE( host_res )
static struct platform_device *myDevices[HOST_NUM];

static int panasonic_ehci_device_init( void )
{
	int i, isSomeone;

	internal_host_peripheryhw_init();
	
	isSomeone=0;
	for( i=0 ; i < HOST_NUM ; i++ ){
		assert( myDevices[i] == NULL );

		myDevices[i] = platform_device_register_simple( "panasonic-ehci", i, (struct resource*)host_res[i], ARRAY_SIZE(host_res[i]) );
		if( IS_ERR(myDevices[i]) ){
			printk( "%s:%d error i=%d\n", __FUNCTION__, __LINE__, i );
			myDevices[i]=NULL;
			continue;
		}
		myDevices[i]->dev.coherent_dma_mask=0xffffffffull;
		myDevices[i]->dev.dma_mask = &myDevices[i]->dev.coherent_dma_mask;
		isSomeone=1;
	}
	
	if( isSomeone ) return 0;
	return -1;
}

static void panasonic_ehci_device_exit( void )
{
	int i;
	
	for( i=0 ; i < HOST_NUM ; i++ ){
		if( myDevices[i] == NULL ) continue;
		platform_device_unregister( myDevices[i] );
		myDevices[i] = NULL;
	}
}

#if defined(CONFIG_USB_PANASONIC_TEST_MODE_IN_USB_DRIVER)
#include "ehci-panasonic-test-mode.c"
#endif

